<?php
session_start();
require_once "db.php";

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch user referral code
$stmt = $conn->prepare("SELECT referral_code FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($referral_code);
$stmt->fetch();
$stmt->close();

// Helper function to get team members by level recursively
function getTeamMembers($conn, $user_ids, $level=1, &$members=[]){
    if(empty($user_ids) || $level>3) return;
    $in  = str_repeat('?,', count($user_ids) - 1) . '?';
    $types = str_repeat('i', count($user_ids));
    $stmt = $conn->prepare("SELECT id, phone FROM users WHERE referred_by IN ($in)");
    $stmt->bind_param($types, ...$user_ids);
    $stmt->execute();
    $res = $stmt->get_result();
    $next_level_ids = [];
    while($row = $res->fetch_assoc()){
        $row['level'] = $level;
        $members[$level][] = $row;
        $next_level_ids[] = $row['id'];
    }
    $stmt->close();
    if($next_level_ids) getTeamMembers($conn, $next_level_ids, $level+1, $members);
}

// Get team members
$members = [];
getTeamMembers($conn, [$user_id], 1, $members);

// Compute stats
$total_members = 0;
$active_members = 0;
$all_team_user_ids = [];

foreach($members as $level=>$users){
    foreach($users as $key => $u){
        $total_members++;
        $all_team_user_ids[] = $u['id'];

        // check active investments
        $stmt = $conn->prepare("SELECT COUNT(*) FROM investments WHERE user_id=? AND status='active'");
        $stmt->bind_param("i",$u['id']);
        $stmt->execute();
        $stmt->bind_result($active_count);
        $stmt->fetch();
        $stmt->close();
        if($active_count>0) $active_members++;

        // mark status for tab display
        $members[$level][$key]['status'] = ($active_count>0)? 'Active':'Inactive';
    }
}

// Total and Today's Income *for bonuses YOU received*
// referral_bonuses table: id,user_id,from_user_id,to_user_id,level,amount,created_at
$stmt = $conn->prepare("SELECT SUM(amount) FROM referral_bonuses WHERE to_user_id=?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($total_income);
$stmt->fetch();
$stmt->close();
$total_income = $total_income ?: 0;

// Today's income
$today = date('Y-m-d');
$stmt = $conn->prepare("SELECT SUM(amount) FROM referral_bonuses WHERE to_user_id=? AND DATE(created_at)=?");
$stmt->bind_param("is", $user_id, $today);
$stmt->execute();
$stmt->bind_result($today_income);
$stmt->fetch();
$stmt->close();
$today_income = $today_income ?: 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Team</title>
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<style>
body {margin:0;font-family:Arial,sans-serif;background:url("car.jpg") no-repeat center center fixed;background-size:cover;}
.overlay {background: rgba(0,0,0,0.55); min-height:100vh; display:flex; flex-direction:column;}
.header {color:#fff;text-align:center;padding:14px;font-size:21px;font-weight:bold;position:fixed;top:0;left:0;right:0;background: rgba(0,0,0,0.6);z-index:1000;}
.content {padding:12px;margin-top:70px;margin-bottom:140px;}
.ref-box,.stat-box,.level-box,.tab-content{background:#fff;border-radius:12px;padding:16px;margin-bottom:18px;box-shadow:0 4px 10px rgba(0,0,0,0.2);}
.ref-box h3,.level-box h3{text-align:center;color:#444;margin:0 0 10px 0;font-size:15px;}
.ref-item,.level-item{display:flex;align-items:center;justify-content:space-between;padding:8px 0;border-bottom:1px solid #eee;font-size:14px;color:#333;}
.ref-item:last-child,.level-item:last-child{border-bottom:none;}
.ref-label{font-size:13px;color:#666;}
.ref-value{font-size:13px;color:#2e7d32;font-weight:bold;word-break:break-all;}
.ref-item button{padding:6px 12px;border:none;border-radius:8px;background:#2e7d32;color:#fff;font-size:12px;cursor:pointer;}
.ref-item button:hover{background:#256428;}
.stats-container{display:grid;grid-template-columns:1fr 1fr;gap:12px;margin-bottom:18px;}
.stat-box h4{margin:0;font-size:14px;color:#666;}
.stat-box .value{margin-top:6px;font-size:18px;font-weight:bold;color:#2e7d32;}
.level-item span{font-weight:bold;color:#2e7d32;}
.tabs{display:flex;justify-content:space-around;margin-bottom:10px;}
.tab{flex:1;padding:10px;text-align:center;background:#eee;cursor:pointer;font-size:14px;font-weight:bold;border-radius:6px 6px 0 0;color:#444;}
.tab.active{background:#2e7d32;color:#fff;}
.tab-content{border-radius:0 0 12px 12px;padding:12px;}
.phone{padding:8px 0;border-bottom:1px solid #eee;font-size:14px;color:#333;}
.phone:last-child{border-bottom:none;}
.bottom-nav{position:fixed;bottom:0;left:0;right:0;background:#fff;display:flex;justify-content:space-around;border-top:1px solid #ddd;z-index:1000;}
.bottom-nav a{flex:1;text-align:center;padding:8px 0;color:#555;font-size:12px;text-decoration:none;}
.bottom-nav a i{display:block;font-size:18px;margin-bottom:3px;}
.bottom-nav a.active{color:#2e7d32;font-weight:bold;}
.status-label{font-size:12px;padding:2px 6px;border-radius:6px;color:#fff;}
.active-status{background:#2e7d32;}
.inactive-status{background:#888;}
</style>
</head>
<body>
<div class="overlay">
  <div class="header">Team</div>
  <div class="content">
    <!-- Referral Box -->
    <div class="ref-box">
      <h3>Your Referral Details</h3>
      <div class="ref-item">
        <div class="ref-label">Referral Code</div>
        <div class="ref-value"><?php echo htmlspecialchars($referral_code); ?></div>
        <button onclick="copyText('<?php echo $referral_code; ?>')">Copy Code</button>
      </div>
      <div class="ref-item">
        <div class="ref-label">Referral Link</div>
        <div class="ref-value" id="refLink">https://example.com/register?ref=<?php echo urlencode($referral_code); ?></div>
        <button onclick="copyText(document.getElementById('refLink').innerText)">Copy Link</button>
      </div>
    </div>

    <!-- Stats -->
    <div class="stats-container">
      <div class="stat-box">
        <h4>Total Team Members</h4>
        <div class="value"><?php echo $total_members; ?></div>
      </div>
      <div class="stat-box">
        <h4>Active Members</h4>
        <div class="value"><?php echo $active_members; ?></div>
      </div>
      <div class="stat-box">
        <h4>Total Team Income</h4>
        <div class="value">TZS <?php echo number_format($total_income,2); ?></div>
      </div>
      <div class="stat-box">
        <h4>Today's Income</h4>
        <div class="value">TZS <?php echo number_format($today_income,2); ?></div>
      </div>
    </div>

    <!-- Levels -->
    <div class="level-box">
      <h3>Team Commission Levels</h3>
      <div class="level-item">Level 1 <span>25%</span></div>
      <div class="level-item">Level 2 <span>5%</span></div>
      <div class="level-item">Level 3 <span>1%</span></div>
    </div>

    <!-- Member Tabs -->
    <div class="tabs">
      <div class="tab active" onclick="showTab(1)">Level 1</div>
      <div class="tab" onclick="showTab(2)">Level 2</div>
      <div class="tab" onclick="showTab(3)">Level 3</div>
    </div>

    <?php for($lvl=1;$lvl<=3;$lvl++): ?>
      <div class="tab-content" id="tab<?php echo $lvl; ?>" style="<?php echo ($lvl==1)?'display:block':'display:none';?>">
        <?php if(isset($members[$lvl])): ?>
          <?php foreach($members[$lvl] as $m): ?>
            <div class="phone">
              <?php echo htmlspecialchars($m['phone']); ?>
              <span class="status-label <?php echo ($m['status']=='Active')?'active-status':'inactive-status'; ?>">
                <?php echo $m['status']; ?>
              </span>
            </div>
          <?php endforeach; ?>
        <?php else: ?>
          <div class="phone">No members</div>
        <?php endif; ?>
      </div>
    <?php endfor; ?>

  </div>

  <div class="bottom-nav">
    <a href="home.php"><i class="fas fa-home"></i>Home</a>
    <a href="collect.php"><i class="fas fa-coins"></i>Collect</a>
    <a href="device.php"><i class="fas fa-microchip"></i>Device</a>
    <a href="team.php" class="active"><i class="fas fa-users"></i>Team</a>
    <a href="me.php"><i class="fas fa-user"></i>Me</a>
  </div>
</div>

<script>
function copyText(text){
    navigator.clipboard.writeText(text).then(()=>{ alert("Copied: "+text); });
}
function showTab(level){
    document.querySelectorAll('.tab').forEach((el,idx)=>el.classList.toggle('active', idx===level-1));
    document.querySelectorAll('.tab-content').forEach((el,idx)=>el.style.display=(idx===level-1)?'block':'none');
}
</script>
</body>
</html>
