<?php
session_start();
require_once "db.php"; // lazima iwe na $conn

// ====== Check kama user ame-login ======
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}
$user_id = $_SESSION['user_id'];

// ====== Fetch user balance ======
$sql = "SELECT balance, password FROM users WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
if ($result->num_rows < 1) {
    die("User not found.");
}
$user = $result->fetch_assoc();
$balance = $user['balance'];
$hashed_password = $user['password'];

// ====== Handle withdraw form ======
$success_msg = $error_msg = "";
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $mainnet = $_POST['mainnet'] ?? '';
    $address = trim($_POST['address'] ?? '');
    $amount = floatval($_POST['amount'] ?? 0);
    $password = $_POST['password'] ?? '';

    // Validate mainnet selection
    $allowed_mainnets = ['BEP20-USDT','TRC20-USDT','POLYGON-USDC','TRX'];
    if (!in_array($mainnet, $allowed_mainnets)) {
        $error_msg = "Please select a valid mainnet.";
    } elseif ($amount <= 0 || $amount > $balance) {
        $error_msg = "Invalid amount.";
    } elseif (!password_verify($password, $hashed_password)) {
        $error_msg = "Invalid security password.";
    } elseif (empty($address)) {
        $error_msg = "Withdrawal address is required.";
    } else {
        // Calculate fees (10%)
        $fee = round($amount * 0.10, 2);
        $taxes = $fee; 
        $actual_arrival = $amount - $fee - $taxes;

        if ($actual_arrival <= 0) {
            $error_msg = "Amount too small after fees.";
        } else {
            // Deduct balance
            $new_balance = $balance - $amount;
            $update_sql = "UPDATE users SET balance = ? WHERE id = ?";
            $stmt_up = $conn->prepare($update_sql);
            $stmt_up->bind_param("di", $new_balance, $user_id);
            $stmt_up->execute();

            // Insert withdrawal record
            $ins_sql = "INSERT INTO usdt_withdrawals (user_id, mainnet, address, amount, fee, taxes, actual_arrival, status, created_at, updated_at) 
                        VALUES (?,?,?,?,?,?,?,'pending',NOW(),NOW())";
            $stmt_ins = $conn->prepare($ins_sql);
            $stmt_ins->bind_param("issdddd", $user_id, $mainnet, $address, $amount, $fee, $taxes, $actual_arrival);
            if ($stmt_ins->execute()) {
                $success_msg = "Withdrawal request submitted successfully.";
                $balance = $new_balance; // update display
            } else {
                $error_msg = "Error saving withdrawal request.";
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width,initial-scale=1"/>
  <title>Smart360 - Withdraw</title>
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800&display=swap');
    :root{
      --bg1:#1ca68f;
      --bg2:#2aa78d;
      --accent:#1fbf9a;
      --accent-dark:#16826f;
    }
    *{margin:0;padding:0;box-sizing:border-box}
    body{
      font-family:Inter,system-ui;
      background:linear-gradient(180deg,var(--bg1),var(--bg2));
      display:flex;justify-content:center;
      padding:24px 12px;
      color:#fff;
    }
    .phone{
      width:390px;min-height:780px;
      background:rgba(255,255,255,0.02);
      border-radius:28px;
      padding:18px 16px 98px;
      box-shadow:0 18px 40px rgba(5,40,35,0.45);
      position:relative;
    }
    h1{text-align:center;font-size:22px;margin-bottom:20px;font-weight:700}
    .msg{padding:10px;margin-bottom:15px;border-radius:8px;font-size:14px;text-align:center}
    .msg.success{background:rgba(0,255,100,0.2);color:#0f0}
    .msg.error{background:rgba(255,0,0,0.2);color:#f88}

    .balance-box{
      background:linear-gradient(180deg,var(--accent),var(--accent-dark));
      padding:20px;
      border-radius:16px;
      text-align:center;
      margin-bottom:20px;
    }
    .balance-box h2{font-size:14px;font-weight:400;margin-bottom:8px}
    .balance-box .amount{font-size:24px;font-weight:700}

    .section{margin-bottom:20px}
    .section h3{font-size:14px;font-weight:600;margin-bottom:10px;color:var(--accent)}
    
    .options{display:flex;flex-wrap:wrap;gap:8px}
    .option{
      flex:1 1 45%;
      background:rgba(0,0,0,0.25);
      padding:10px;
      border-radius:10px;
      text-align:center;
      font-size:14px;
      font-weight:600;
      cursor:pointer;
      border:1.5px solid transparent;
      user-select:none;
    }
    .option.active{
      background:rgba(255,255,255,0.1);
      border-color:var(--accent);
      color:var(--accent);
    }

    .input-box{
      background:rgba(0,0,0,0.25);
      padding:12px;
      border-radius:10px;
      margin-bottom:14px;
    }
    .input-box input{
      width:100%;
      background:transparent;
      border:none;
      outline:none;
      color:#fff;
      font-size:14px;
    }
    .input-box input::placeholder{color:#bbb}

    .details{font-size:13px;margin-top:10px;line-height:1.6}
    .details span{color:var(--accent);font-weight:600}

    .btn{
      display:block;
      width:100%;
      padding:14px;
      border-radius:12px;
      border:none;
      cursor:pointer;
      font-size:15px;
      font-weight:700;
      background:linear-gradient(180deg,var(--accent),var(--accent-dark));
      color:#fff;
      margin-top:16px;
    }

    .reminder{
      background:rgba(0,0,0,0.25);
      padding:16px;
      border-radius:12px;
      font-size:13px;
      line-height:1.6;
      margin-top:24px;
    }
    .reminder h4{font-size:14px;font-weight:700;color:var(--accent);margin-bottom:8px}
    .reminder ul{padding-left:18px;margin-bottom:10px}
    .reminder li{margin-bottom:4px}
  </style>
  <script>
    function calcFees(){
      let amount = parseFloat(document.getElementById("amount").value)||0;
      let fee = (amount*0.10).toFixed(2);
      let taxes = fee;
      let arrival = (amount - fee - taxes).toFixed(2);
      document.getElementById("fee").innerText = fee+" USDT";
      document.getElementById("taxes").innerText = taxes+" USDT";
      document.getElementById("arrival").innerText = (arrival>0?arrival:0)+" USDT";
    }

    function selectOption(el, value){
      document.querySelectorAll('.option').forEach(opt => opt.classList.remove('active'));
      el.classList.add('active');
      document.getElementById("mainnet").value = value;
    }

    function validateForm(e){
      let mainnet = document.getElementById("mainnet").value;
      if(mainnet === ""){
        alert("Please select a mainnet first.");
        e.preventDefault();
        return false;
      }
    }
  </script>
</head>
<body>
  <div class="phone">
    <h1>Withdraw</h1>

    <?php if($success_msg): ?><div class="msg success"><?= $success_msg ?></div><?php endif; ?>
    <?php if($error_msg): ?><div class="msg error"><?= $error_msg ?></div><?php endif; ?>

    <!-- Balance -->
    <div class="balance-box">
      <h2>Currently available assets (USDT)</h2>
      <div class="amount"><?= number_format($balance,6) ?></div>
    </div>

    <form method="post" onsubmit="validateForm(event)">
      <!-- Select Mainnet -->
      <div class="section">
        <h3>Select Mainnet</h3>
        <div class="options">
          <div class="option" onclick="selectOption(this,'BEP20-USDT')">BEP20-USDT</div>
          <div class="option" onclick="selectOption(this,'TRC20-USDT')">TRC20-USDT</div>
          <div class="option" onclick="selectOption(this,'POLYGON-USDC')">POLYGON-USDC</div>
          <div class="option" onclick="selectOption(this,'TRX')">TRX</div>
        </div>
        <input type="hidden" name="mainnet" id="mainnet" value="">
      </div>

      <!-- Withdrawal address -->
      <div class="section">
        <h3>Withdrawal address</h3>
        <div class="input-box">
          <input type="text" name="address" placeholder="Please enter or paste the wallet address" required/>
        </div>
      </div>

      <!-- Withdrawal amount -->
      <div class="section">
        <h3>Withdrawal amount</h3>
        <div class="input-box">
          <input type="number" step="0.01" min="1" id="amount" name="amount" placeholder="Please enter the transfer amount" oninput="calcFees()" required/>
        </div>
        <div class="details">
          Minimum withdrawal amount: <span>1.00 USDT</span><br>
          Maximum withdrawal amount: <span>99999999.00 USDT</span>
        </div>
      </div>

      <!-- Security password -->
      <div class="section">
        <h3>Security password</h3>
        <div class="input-box">
          <input type="password" name="password" placeholder="Security password" required/>
        </div>
      </div>

      <!-- Fees & Info -->
      <div class="details">
        Fee: <span id="fee">0 USDT</span><br>
        Taxes: <span id="taxes">0 USDT</span><br>
        Actual arrival: <span id="arrival">0 USDT</span>
      </div>

      <!-- Confirm Button -->
      <button type="submit" class="btn">Confirm</button>
    </form>

    <!-- Warm Reminder -->
    <div class="reminder">
      <h4>Warm reminder</h4>
      <p><strong>Support:</strong></p>
      <ul>
          <li>BEP-20 (USDT)</li>
          <li>BEP-20 (USDC)</li>
          <li>TRC-20 (USDT)</li>
      </ul>
      <p>The minimum withdrawal amount is <span>1 USDT</span>.</p>
      <p>If you have any questions, please contact online customer service.</p>
    </div>
  </div>
</body>
</html>
